<?php

/**
 * The public-facing functionality of the plugin.
 */
class Qls_Shipment_Public
{
    private Qls_Shipment_Cart $qls_shipment_cart;
    private Qls_Shipment_Checkout $qls_shipment_checkout;

    public function __construct(private string $plugin_name, private string $version)
    {
        $this->qls_shipment_cart = new Qls_Shipment_Cart($plugin_name, $version);
        $this->qls_shipment_checkout = new Qls_Shipment_Checkout($plugin_name, $version);
    }

    public function enqueue_styles(): void
    {
        if (!is_checkout()) {
            return;
        }

        wp_enqueue_style(
            $this->plugin_name,
            plugins_url('/../build/public.css', __FILE__),
            [],
            filemtime(dirname(__FILE__) . '/../build/public.css')
        );
    }

    public function enqueue_scripts(): void
    {
        if (!is_checkout()) {
            return;
        }

        $script_path = dirname(__FILE__) . '/../build/public.js';
        $script_asset_path = dirname(__FILE__) . '/../build/public.asset.php';
        $script_asset = file_exists($script_asset_path)
            ? require($script_asset_path)
            : array('dependencies' => [], 'version' => filemtime($script_path));
        $script_url = plugins_url('/../build/public.js', __FILE__);

        wp_enqueue_script(
            $this->plugin_name,
            $script_url,
            $script_asset['dependencies'],
            $script_asset['version'],
        );
    }

    public function define_hooks(Qls_Shipment_Loader $loader): void
    {
        $this->qls_shipment_cart->define_hooks($loader);
        $this->qls_shipment_checkout->define_hooks($loader);

        $loader->add_action('wp_enqueue_scripts', $this, 'enqueue_styles');
        $loader->add_action('wp_enqueue_scripts', $this, 'enqueue_scripts');
        $loader->add_action('woocommerce_before_order_notes', $this, 'display_qls_shipment_methods');

        $loader->add_filter('woocommerce_locate_template', $this, 'add_qls_shipment_methods_template', 5, 2);
        $loader->add_filter('woocommerce_locate_template', $this, 'overwrite_cart_shipping_template', 15, 2);
        $loader->add_filter('woocommerce_update_order_review_fragments', $this, 'render_shipment_methods');
        $loader->add_filter('woocommerce_order_shipping_to_display', $this, 'add_options_to_shipping_render', 10, 2);
    }

    /**
     * Display the shipping methods in the checkout
     *
     * @return void
     */
    public function display_qls_shipment_methods(): void
    {
        echo $this->qls_shipment_methods_template();
    }

    /**
     * Get the shipment method template as string
     *
     * @return string
     */
    protected function qls_shipment_methods_template(): string
    {
        ob_start();
        wc_get_template('checkout/qls-shipment-methods.php', [
            'shipping_methods' => $this->qls_shipment_cart->get_qls_shipping_methods(),
            'is_shipping_available' => $this->qls_shipment_cart->is_shipping_available(),
            'are_shipping_methods_customer_address_based' => $this->qls_shipment_cart->are_shipping_methods_customer_address_based(),
        ]);
        return ob_get_clean();
    }

    /**
     * Add the ability to override the qls-shipment-methods template
     *
     * @param $template
     * @param $template_name
     * @return string
     */
    public function add_qls_shipment_methods_template($template, $template_name): string
    {
        if ($template_name !== 'checkout/qls-shipment-methods.php') {
            return $template;
        }

        return __DIR__ . '/partials/qls-shipment-methods.php';
    }

    /**
     * Overwrite the default cart-shipping template
     *
     * @param $template
     * @param $template_name
     * @return string
     */
    public function overwrite_cart_shipping_template($template, $template_name): string
    {
        if ($template_name !== 'cart/cart-shipping.php') {
            return $template;
        }

        return __DIR__ . '/partials/cart-shipping.php';
    }

    /**
     * Add the shipping methods to the fragments when the order is updated
     *
     * @param $fragments
     * @return array
     */
    public function render_shipment_methods($fragments): array
    {
        $fragments['.qls-shipment-methods'] = $this->qls_shipment_methods_template();

        return $fragments;
    }

    /**
     * Display the chosen shipping method
     *
     * @param $shipping
     * @param WC_Order $order
     * @return string
     */
    public function add_options_to_shipping_render($shipping, WC_Order $order): string
    {
        $shipping_methods = $order->get_shipping_methods();

        if (count($shipping_methods) !== 1) {
            return $shipping;
        }

        $shipping_line = array_values($shipping_methods)[0];

        foreach ($shipping_line->get_meta('_qls_options') as $label) {
            $shipping .= '<br><strong>' . $label . ':</strong> ' . $shipping_line->get_meta($label);
        }

        return $shipping;
    }

}
