<?php

/**
 * Contains all functionality related to the shipping method handling during checkout
 */
class Qls_Shipment_Checkout
{

    public function __construct(private string $plugin_name, private string $version)
    {
    }

    public function define_hooks(Qls_Shipment_Loader $loader): void
    {
        $loader->add_action('woocommerce_checkout_create_order_shipping_item', $this, 'add_metadata_to_qls_shipment_line', 10, 2);
        $loader->add_action('woocommerce_checkout_create_order', $this, 'add_metadata_to_order', 10, 2);
    }

    public function add_metadata_to_order(WC_Order $order, array $data): void
    {
        $qls_shipping_methods = WC()->session->get('qls_shipping_methods');
        $chosen_shipping_methods = WC()->session->get('chosen_shipping_methods');

        foreach ($chosen_shipping_methods as $chosen_method) {
            $qls_shipping_method = array_values(
                array_filter($qls_shipping_methods, static function ($method) use ($chosen_method) {
                    return "qls-shipment-{$method['id']}" === $chosen_method;
                })
            )[0];

            foreach ($qls_shipping_method['options'] as $key => $option) {
                // Data is set in class-qls-shipment-cart::update_checkout_data()
                $session_key = WC()->session->get("chosen_qls-shipment-method-{$qls_shipping_method['id']}-option-{$key}");
                $order->add_meta_data("_qls_{$key}", $session_key);
            }
        }
    }


    /**
     * Save the chosen shipping method data to the order
     *
     * @param WC_Order_Item_Shipping $item
     * @param $package_key
     * @return void
     */
    public function add_metadata_to_qls_shipment_line(WC_Order_Item_Shipping $item, $package_key): void
    {
        // Get the chosen shipping method for this package.
        $chosen_shipment_methods = WC()->session->get('chosen_shipping_methods');
        $chosen_shipment_method = $chosen_shipment_methods[$package_key];

        // Get the options fetched from the API belonging to the chosen shipping method.
        $qls_shipping_methods = WC()->session->get('qls_shipping_methods');
        $qls_shipping_method = array_values(array_filter($qls_shipping_methods, fn($method) => 'qls-shipment-' . $method['id'] === $chosen_shipment_method))[0];

        // Add the tags to the order
        $item->add_meta_data('_tags', $qls_shipping_method['tags']);

        foreach ($qls_shipping_method['options'] as $key => $option) {
            $session_key = WC()->session->get('chosen_qls-shipment-method-' . $qls_shipping_method['id'] . '-option-' . $key);

            // Add readable data to the order
            $item->add_meta_data($option['label'], $option['values'][$session_key]);

            // Add data to the order to retrieve from the API
            $item->add_meta_data('_qls_' . $key, $session_key);
        }

        // Add the available options key and label to the order
        $item->add_meta_data('_qls_options', array_map(fn($option) => $option['label'], $qls_shipping_method['options']));

        $item->save_meta_data();
    }

}
