<?php

/**
 * Contains all functionality related to the shipping method calculation in the cart
 */
class Qls_Shipment_Cart
{

    private Qls_Shipment_Api $api;

    public function __construct(private string $plugin_name, private string $version)
    {
        // Initialize the API using the plugin settings
        $this->api = new Qls_Shipment_Api(
            get_option('qls_shipment_api_endpoint'),
            get_option('qls_shipment_api_username'),
            get_option('qls_shipment_api_password'),
            get_option('qls_shipment_integration_id'),
        );

    }

    public function define_hooks(Qls_Shipment_Loader $loader): void
    {
        $loader->add_action('init', $this, 'set_shipping_method_count_transient');
        $loader->add_action('woocommerce_checkout_update_order_review', $this, 'update_checkout_data');
        $loader->add_filter('woocommerce_package_rates', $this, 'qls_package_rates');
    }

    /**
     * Overwrite the shipping method count transients set by WooCommerce to fool WooCommerce into thinking that the
     * cart needs shipping (WC_Cart::needs_shipping())
     *
     * @return void
     */
    public function set_shipping_method_count_transient(): void
    {
        $transient_version = WC_Cache_Helper::get_transient_version('shipping');
        $transient_names = ['wc_shipping_method_count_legacy', 'wc_shipping_method_count'];

        foreach ($transient_names as $transient_name) {
            $transient_value = array(
                'version' => $transient_version,
                'value' => 1,
            );

            set_transient($transient_name, $transient_value, DAY_IN_SECONDS * 30);
        }
    }

    /**
     * Get the QLS Shipping Methods set when determining the package rates from the session
     *
     * @return array|null
     */
    public function get_qls_shipping_methods(): array|null
    {
        return WC()->session->get('qls_shipping_methods');
    }

    /**
     * Check if the shipping methods are based on the customer address
     *
     * @return bool
     */
    public function are_shipping_methods_customer_address_based(): bool
    {
        return WC()->session->get('qls_shipping_methods_method') === 'customer';
    }

    public function qls_package_rates(): array
    {
        // When the customer address is not known and no fallback address is available, return the fallback rates.
        if (!$this->customer_address_known() && !$this->has_fallback_address()) {
            return $this->get_fallback_rate();
        }

        $address_overwrite = null;

        if (!$this->customer_address_known() && $this->has_fallback_address()) {
            $address_overwrite = [
                "is_company" => false,
                "address1" => get_option('woocommerce_store_address', ''),
                "address2" => get_option('woocommerce_store_address_2', ''),
                "zipcode" => get_option('woocommerce_store_postcode', ''),
                "city" => get_option('woocommerce_store_city', ''),
                'country' => get_option('woocommerce_default_country', 'US:CA'),
            ];
            WC()->session->set('qls_shipping_methods_method', 'base_address');
        } else {
            WC()->session->set('qls_shipping_methods_method', 'customer');
        }

        // Fetch the shipping methods from the QLS API
        $shipping_methods = $this->api->get_shipping_methods($address_overwrite);

        // Save the shipping methods to the session, so we don't need to fetch them again and make sure that we have the
        // same shipping methods available used to determine the rates.
        WC()->session->set('qls_shipping_methods', $shipping_methods);

        // When no shipping methods are returned by the API, use the fallback rate.
        if (!count($shipping_methods ?? [])) {
            return $this->get_fallback_rate();
        }

        $rates = [];

        foreach ($shipping_methods as $shipping_method) {
            $id = 'qls-shipment-' . $shipping_method['id'];

            $rates[$id] = new WC_Shipping_Rate(
                $id,
                $shipping_method['title'],
                $shipping_method['price_excl'],
                WC_Tax::calc_shipping_tax($shipping_method['price_excl'], WC_Tax::get_shipping_tax_rates()),
                $id
            );
        }

        return $rates;
    }

    /**
     * Determines if all required fields are set for shipping to be determined using the customer address.
     *
     * @return bool
     */
    protected function customer_address_known(): bool
    {
        return WC()->customer->get_shipping_address_1() && WC()->customer->get_shipping_postcode() && WC()->customer->get_shipping_city() && WC()->customer->get_shipping_country();
    }

    /**
     * Check if the store is set to use the store base address as fallback.
     *
     * @return bool
     */
    protected function has_fallback_address(): bool
    {
        return get_option('woocommerce_default_customer_address', 'base') === 'base';
    }

    /**
     * Determines if the shipping costs can be determined.
     *
     * @return bool
     */
    public function is_shipping_available(): bool
    {
        return $this->customer_address_known() || $this->has_fallback_address();
    }

    /**
     * @return array|WC_Shipping_Rate[]
     */
    protected function get_fallback_rate(): array
    {
        $fallback = get_option('qls_shipment_fallback_price');

        if (!$fallback) {
            return [];
        }

        $tax = WC_Tax::calc_tax($fallback, WC_Tax::get_shipping_tax_rates(), true);
        $fallback_excl = $fallback - array_sum($tax);

        return [
            'qls-shipment-fallback' => new WC_Shipping_Rate(
                'qls-shipment-fallback',
                __('Shipment', 'qls-shipment'),
                $fallback_excl,
                WC_Tax::calc_shipping_tax($fallback_excl, WC_Tax::get_shipping_tax_rates()),
                'fallback_price'
            )
        ];
    }

    /**
     * Save the QLS shipping data to the session when the checkout is updated.
     *
     * @param $postdata
     * @return void
     */
    public function update_checkout_data($postdata): void
    {
        $data = [];
        parse_str($postdata, $data);

        if (isset($data['qls_shipment_method'])) {
            WC()->session->set('chosen_shipping_methods', ['qls-shipment-' . $data['qls_shipment_method']]);
        }

        foreach ($data as $key => $value) {
            // Save all option data to the session
            if (str_starts_with($key, 'qls-shipment-method-')) {
                WC()->session->set('chosen_' . $key, $value);
            }
        }
    }

}
