<?php

/**
 * Contains the API connection to QLS
 */
class Qls_Shipment_Api
{

    public function __construct(protected string $api_url, protected string $api_username, protected string $api_password, protected string $shop_integration_id)
    {
    }

    protected function number_format($price): float
    {
        return (float)number_format($price, wc_get_price_decimals());
    }

    /**
     * Make a request to the QLS API with the correct authorization header
     *
     * @param $url
     * @param array $body
     * @return array|WP_Error
     */
    protected function make_request($url, array $body = []): array|WP_Error
    {
        return wp_remote_post($url, [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($this->api_username . ':' . $this->api_password),
                'Content-Type' => 'application/json',
            ],
            'body' => json_encode($body),
        ]);
    }

    public function usesCheckoutService(): bool
    {
        $host = parse_url($this->api_url, PHP_URL_HOST);

        return in_array($host, ['checkout.test.qls.nl', 'checkout.qls.nl']);
    }
    /**
     * Checks if the current settings allow to access the API.
     *
     * @return bool
     */
    public function check_authenticated(): bool
    {
        $healthUrl = $this->api_url . '/companies';

        if ($this->usesCheckoutService()) {
            $id = get_option('qls_shipment_integration_id');
            $healthUrl = sprintf($this->api_url, $id).'/health';
        }

        try {
            $return_data = $this->make_request($healthUrl);
            if (is_wp_error($return_data)) {
                return false;
            }
        } catch (Exception $e) {
            return false;
        }

        $response = json_decode($return_data['body'], true);
        return $response['meta']['code'] !== 401;
    }

    /**
     * Get the available shipping methods from the QLS API.
     * @param array|null $address_overwrite Address to use for the API call (defaults to the customer address)
     *
     * @return array|null
     */
    public function get_shipping_methods(array|null $address_overwrite = null): null|array
    {
        $url = $this->api_url . 'external-services/woocommerce/' . $this->shop_integration_id . '/shipping-methods';

        if ($this->usesCheckoutService()) {
            $url = sprintf($this->api_url, $this->shop_integration_id);
        }

        $body = $this->get_data_from_cart($address_overwrite);

        $return_data = $this->make_request($url, $body);

        if ($return_data instanceof WP_Error) {
            Qls_Shipment_Logger::error(
                "Error occurred when getting shipment methods from the API. \n" .
                'URL: ' . $url . "\n" .
                'Request: ' . print_r($body, true) . "\n" .
                'Error: ' . $return_data->get_error_message()
            );
            return null;
        }

        $response = json_decode($return_data['body'], true);

        if (isset($response['meta']['code']) && $response['meta']['code'] !== 200) {
            Qls_Shipment_Logger::error(
                "Error occurred when getting shipment methods from the API. \n" .
                'URL: ' . $url . "\n" .
                'Request: ' . print_r($body, true) . "\n" .
                'Error: ' . $return_data['body']
            );
            return null;
        }

        if (!isset($response['data']) || count($response['data']) === 0) {
            Qls_Shipment_Logger::warning('No shipment methods were returned from the API');
            return null;
        }

        return array_map(function (array $method) {
            // Generate an ID for the shipping method. Base the ID on the tags and the keys of the available options.
            $id = md5(implode('', array_merge($method['tags'], array_keys($method['options'] ?? []))));

            // Calculate tax for the shipping method based on the including price returned from the API and the shipping tax rates.
            $tax = WC_Tax::calc_tax($method['price_incl'], WC_Tax::get_shipping_tax_rates(), true);

            // Remove the calculated taxes from the including price.
            $price_excl = $method['price_incl'] - array_sum($tax);

            return array_merge($method, compact('id', 'price_excl', 'tax'));
        }, $response['data']);
    }

    /**
     * Get the API request body for QLS based on the cart contents. If the $address_overwrite is specified, this
     * address is used instead of the address the is specified in the cart.
     * @param array|null $address_overwrite Address to use for the API call (defaults to the customer address)
     *
     * @return array
     */
    protected function get_data_from_cart(array|null $address_overwrite = null): array
    {
        $cart = WC()->cart;

        // Forcefully calculate totals without shipping
        $this->force_calculate_totals_without_shipping($cart);

        // Check if one of the coupons grants free shipping
        $has_free_shipping_coupon = $this->has_free_shipping_coupon($cart);

        // If no overwrite is specified, get the address from the cart
        $shipping_address = $address_overwrite ?? [
            "is_company" => !empty($cart->get_customer()->get_shipping_company()) ? 1 : 0,
            "address1" => $cart->get_customer()->get_shipping_address_1(),
            "address2" => $cart->get_customer()->get_shipping_address_2(),
            "zipcode" => $cart->get_customer()->get_shipping_postcode(),
            "city" => $cart->get_customer()->get_shipping_city(),
            "country" => $cart->get_customer()->get_shipping_country()
        ];

        $data = [
            "shipping_address" => $shipping_address,
            "cart" => [
                "price_incl" => $this->number_format((float)WC()->cart->total - WC()->cart->get_shipping_total() - WC()->cart->get_shipping_tax()),
                "price_excl" => $this->number_format((float)WC()->cart->total - $cart->get_total_tax() - WC()->cart->get_shipping_total() - WC()->cart->get_shipping_tax()),
                "products" => $this->get_product_data($cart),
                "free_shipping_coupon" => $has_free_shipping_coupon
            ]
        ];

        return apply_filters('qls_shipment_api_request_data', $data, $cart);
    }

    /**
     * Calculate the cart total, excluding the shipping costs.
     *
     * @param WC_Cart $cart
     * @return void
     */
    protected function force_calculate_totals_without_shipping(WC_Cart $cart): void
    {
        add_filter('woocommerce_cart_ready_to_calc_shipping', '__return_false', 99);
        new WC_Cart_Totals($cart);
        remove_filter('woocommerce_cart_ready_to_calc_shipping', '__return_false', 99);
    }

    /**
     * Determine if a coupon grants free shipping
     *
     * @param WC_Cart $cart
     * @return bool
     */
    public function has_free_shipping_coupon(WC_Cart $cart): bool
    {
        $has_free_shipping = count(array_filter(array_map(function (WC_Coupon $coupon) {
                return $coupon->get_free_shipping();
            }, $cart->get_coupons()))) > 0;

        return apply_filters('qls_shipment_api_has_free_shipping_coupon', $has_free_shipping, $cart);
    }

    /**
     * Get the product data
     *
     * @param WC_Cart $cart
     * @return array
     */
    public function get_product_data(WC_Cart $cart): array
    {
        $product_data = array_values(array_filter(array_map(function (array $item) {
            /** @var WC_Product $product */
            $product = $item['data'];

            $data =  [
                'product_id' => $item['product_id'],
                'variant_id' => $item['variation_id'],
                'title' => $product->get_title(),
                'price_incl' => $this->number_format($item['line_total'] + $item['line_tax']),
                'price_excl' => $this->number_format($item['line_total']),
                'quantity' => $item['quantity'],
                'sku' => $product->get_sku(),
                'length' => $product->get_length(),
                'width' => $product->get_width(),
                'height' => $product->get_height(),
                'weight' => $product->get_weight(),
                "categories" => array_map(function (int $category_id) {
                    return get_term($category_id, 'product_cat')->name;
                }, $product->get_category_ids()),
                "shipping_class" => $product->get_shipping_class()
            ];

            $freeShipmentCalculationMethod = get_option('qls_shipment_free_shipment_calculation_method');

            // other option would be "total" however we do that by default.
            if ($freeShipmentCalculationMethod === 'subtotal') {
                $data['price_incl'] = $this->number_format($item['line_subtotal'] + $item['line_tax']);
            }

            return $data;
        }, $cart->get_cart_contents())));

        return apply_filters('qls_shipment_api_request_products', $product_data, $cart);
    }

}
