<?php

/**
 * The settings page of the plugin.
 */
class Qls_Shipment_SettingsPage
{
    public function __construct(private string $plugin_name, private string $version, private Qls_Shipment_Settings $qls_shipment_settings)
    {
    }

    public function define_hooks(Qls_Shipment_Loader $loader): void
    {
        $loader->add_action('admin_menu', $this, 'add_admin_menu');
        $loader->add_action('admin_init', $this, 'register_fields');

        $loader->add_filter("plugin_action_links_$this->plugin_name/$this->plugin_name.php", $this, 'settings_link');
    }

    /**
     * Add the settings page to the admin menu.
     *
     * @return void
     */
    public function add_admin_menu(): void
    {
        add_options_page(
            __('QLS Shipment Options', 'qls-shipment'),
            __('QLS Shipment Options', 'qls-shipment'),
            apply_filters('qls_shipment_settings_capability', 'manage_options'),
            $this->plugin_name . '-settings',
            [$this, 'display_settings'],
        );
    }

    /**
     * Display the settings page.
     *
     * @return void
     */
    public function display_settings(): void
    {
        // Check if the stored API details provide access to the API
        if ($this->qls_shipment_settings->all_required_settings_set()) {
            $api = new Qls_Shipment_Api(
                get_option('qls_shipment_api_endpoint'),
                get_option('qls_shipment_api_username'),
                get_option('qls_shipment_api_password'),
                get_option('qls_shipment_integration_id'),
            );

            if (!$api->check_authenticated()) {
                ?>
                <div class="notice notice-warning">
                    <p><?= __('It appears that the login details for the QLS API are incorrect. We are unable to communicate to the API.', 'qls-shipment') ?></p>
                </div>
                <?php
            }
        }

        require_once 'partials/' . $this->plugin_name . '-admin-settings-display.php';
    }

    /**
     * Register the settings fields.
     *
     * @return void
     */
    public function register_fields(): void
    {
        add_settings_section(
            'qls-shipment-general',
            null,
            function () {
            },
            $this->plugin_name . '-settings'
        );

        add_settings_field(
            'qls_shipment_api_endpoint',
            __('API Endpoint', 'qls-shipment'),
            [$this, 'render_input'],
            $this->plugin_name . '-settings',
            'qls-shipment-general',
            [
                'type' => 'select',
                'id' => 'qls_shipment_api_endpoint',
                'name' => 'qls_shipment_api_endpoint',
                'required' => true,
                'options' => [
                    'https://api.pakketdienstqls.nl/',
                    'https://api.test.pakketdienstqls.nl/',
                    'https://checkout.qls.nl/api/checkout/woocommerce/%s',
                    'https://checkout.test.qls.nl/api/checkout/woocommerce/%s',
                ],
            ]
        );

        add_settings_field(
            'qls_shipment_api_username',
            __('API Username', 'qls-shipment'),
            [$this, 'render_input'],
            $this->plugin_name . '-settings',
            'qls-shipment-general',
            [
                'type' => 'input',
                'subtype' => 'text',
                'id' => 'qls_shipment_api_username',
                'name' => 'qls_shipment_api_username',
                'required' => true,
            ]
        );

        add_settings_field(
            'qls_shipment_api_password',
            __('API Password', 'qls-shipment'),
            [$this, 'render_input'],
            $this->plugin_name . '-settings',
            'qls-shipment-general',
            [
                'type' => 'input',
                'subtype' => 'password',
                'id' => 'qls_shipment_api_password',
                'name' => 'qls_shipment_api_password',
                'required' => true,
            ]
        );

        add_settings_field(
            'qls_shipment_integration_id',
            __('Integration ID', 'qls-shipment'),
            [$this, 'render_input'],
            $this->plugin_name . '-settings',
            'qls-shipment-general',
            [
                'type' => 'input',
                'subtype' => 'text',
                'id' => 'qls_shipment_integration_id',
                'name' => 'qls_shipment_integration_id',
                'required' => true,
            ]
        );

        add_settings_field(
            'qls_shipment_fallback_price',
            __('Fallback price', 'qls-shipment'),
            [$this, 'render_input'],
            $this->plugin_name . '-settings',
            'qls-shipment-general',
            [
                'type' => 'input',
                'subtype' => 'number',
                'id' => 'qls_shipment_fallback_price',
                'name' => 'qls_shipment_fallback_price',
                'required' => true,
                'min' => 0,
                'step' => 0.01,
            ]
        );

        add_settings_field(
            'qls_shipment_free_shipment_calculation_method',
            __('Shipment price calculation method', 'qls-shipment'),
            [$this, 'render_input'],
            $this->plugin_name . '-settings',
            'qls-shipment-general',
            [
                'type' => 'radio',
                'id' => 'qls_shipment_free_shipment_calculation_method',
                'name' => 'qls_shipment_free_shipment_calculation_method',
                'options' => [
                    'subtotal' => __('Subtotal: before discounts, after tax', 'qls-shipment'),
                    'total' => __('Total: after discounts, after tax', 'qls-shipment')
                ],
                'default' => 'total',
            ]
        );
    }

    /**
     * Render the input field for the setting.
     *
     * @param array $args
     * @return void
     * @throws Exception
     */
    public function render_input(array $args): void
    {
        $raw_value = get_option($args['name']);
        $value = (($args['value_type'] ?? null) == 'serialized') ? serialize($raw_value) : $raw_value;

        switch ($args['type']) {
            case 'input':
                if ($args['subtype'] != 'checkbox') {
                    $prependStart = (isset($args['prepend_value'])) ? '<div class="input-prepend"> <span class="add-on">' . $args['prepend_value'] . '</span>' : '';
                    $prependEnd = (isset($args['prepend_value'])) ? '</div>' : '';
                    $step = (isset($args['step'])) ? 'step="' . $args['step'] . '"' : '';
                    $min = (isset($args['min'])) ? 'min="' . $args['min'] . '"' : '';
                    $max = (isset($args['max'])) ? 'max="' . $args['max'] . '"' : '';

                    echo $prependStart . '<input type="' . $args['subtype'] . '" id="' . $args['id'] . '" "' . $args['required'] . '" ' . $step . ' ' . $max . ' ' . $min . ' name="' . $args['name'] . '" size="40" value="' . esc_attr($value) . '" />' . $prependEnd;
                } else {
                    $checked = ($value) ? 'checked' : '';
                    echo '<input type="' . $args['subtype'] . '" id="' . $args['id'] . '" "' . $args['required'] . '" name="' . $args['name'] . '" size="40" value="1" ' . $checked . ' />';
                }
                break;
            case 'select':
                $prependStart = (isset($args['prepend_value'])) ? '<div class="input-prepend"> <span class="add-on">' . $args['prepend_value'] . '</span>' : '';
                $prependEnd = (isset($args['prepend_value'])) ? '</div>' : '';

                $options = '';

                foreach ($args['options'] as $option) {
                    $options .= '<option value="' . $option . '" ' . ($value === $option ? 'selected' : '') . '>' . $option . '</option>';
                }

                echo $prependStart . '<select id="' . $args['id'] . '" "' . $args['required'] . '" name="' . $args['name'] . '">' . $options . '</select>' . $prependEnd;
                break;

            case 'radio':
                $option_name = $args['name'];
                $option_value = get_option($option_name, $args['default']);

                foreach ($args['options'] as $value => $label) {
                    printf(
                        '<label><input type="radio" name="%s" value="%s" %s> %s</label><br>',
                        esc_attr($option_name),
                        esc_attr($value),
                        checked($option_value, $value, false),
                        esc_html($label)
                    );
                }
                break;
            default:
                throw new Exception();
        }
    }

    /**
     * Add settings link to plugin page
     *
     * @param array $links
     * @return array
     */
    public function settings_link(array $links): array
    {
        $url = esc_url(add_query_arg(
            'page',
            'qls-shipment-settings',
            get_admin_url() . 'options-general.php'
        ));

        $settings_link = "<a href='$url'>" . __('Settings') . '</a>';

        array_unshift($links, $settings_link);

        return $links;
    }

}
